/**
 * @preserve
 * Copyright 2013, Contemporary Control Systems, Inc.
 */

/**
 * Url for XMLHttpRequest() calls.
 *
 * @type {string}
 */
  var xmlCgiUrl = 'cgi-bin/xml-cgi';	  // Execution on BAScontrol.
//var xmlCgiUrl = 'cgi-bin/cgi-rdom';     // Simulation on Zotac server.

/******************************************************************************
 *
 * rdomDoc, now using jQuery.
 *
 ******************************************************************************/

/**
 * Maximum number of bytes in an rdom request. Determined by the size of the
 * recevie buffer on the target.
 *
 *  @define {number} */
var MAX_REQ_SIZE = 512;

/**
 * Creates a new, blank, document.
 *
 * @param {string} docName
 * @param {string} targetUrl
 * @constructor
 */
function rdomDoc(docName, targetUrl)
{
	/** @type {string} */
	this.docName = docName;
	/** @type {string} */
	this.targetUrl = targetUrl;
	/** @type {boolean} */
	this.async = true;
	/** @type {Array.<string>} */
	this.reqList = [];
	/** @type {number} */
	this.reqIdx = 0;
	/** @type {jQuery | null} */
	this.response = null;
	/** @type {function(rdomDoc) | null} */
	this.callback = null;
}

/**
 * Execute AJAX request(s) for XML data from a list of rdom request strings.
 *
 * Uses 'async' member of rdomDoc to do either sync or async request.
 *
 * The rdomDoc.reqList[] array should be loaded with request strings, and the
 * path index should be '0'.
 *
 * The rdomDoc.callback() member should be loaded with the callback function
 * to be executed when the request list is finished.
 *
 * This function can be used to make a single request, or to process a list of
 * requests. If more than one request is given, the children of the second and
 * subsequent requests are appended to the children of the first request.
 */
rdomDoc.prototype.Dispatch = function ()
{
    $.ajax({
        type     : 'POST',
        url      : this.targetUrl,
        data     : this.reqList[this.reqIdx++],
        dataType : 'xml',
        async    : this.async,
        context  : this,
        error    : function (jqXHR, status)
        {
            // Get error to display.
            var errText = 'AJAX Fail:' + status + ' : ' + jqXHR.statusText;
            // Display error.
            alert(errText);
            // Done with this request.
            rdomQ.Done();
        },
        success  : function (data , status , jqXHR)
        {
            /** @type {Element} */
            var docEle;
            /** @type {jQuery} */
            var docXml;

            // Get document: <rdom>.
            docEle = data.documentElement;

            // If 'nak', show error and abort request.
            if ($(docEle).attr('rsp') !== 'ack')
            {
                alert($(docEle).text());
                return;
            }

			// If document has no children (could be 'GetText' or 'SetText' response).
			if ($(docEle).children().length === 0)
			{
				// The repsonse is the document.
				docXml = $(docEle);
			}
			// Else there are children.
			else
			{
				// XML document we need is first (and only) child of <rdom>.
				docXml = $(docEle).children().eq(0);
			}

            // The first response is stored directly. If more than one
            // response, the children of the second and subsequent requests
            // are appended to the children of the first response.
            if (this.response === null)
                this.response = docXml;
            else
                this.response.append(docXml.children());

            // If finished.
            if (this.reqIdx === this.reqList.length)
            {
				// If async mode.
				if (this.async)
				{
					// Send next request.
					rdomQ.Done();
				}
                // Execute callback; may add to Q.
                if (typeof this.callback === 'function')
                    this.callback(this);
				// And return.
                return;
            }
            // Else recursively send next request in this list.
            this.Dispatch();
        }
    });
};

/**
 * @param {string} path
 * @param {boolean} recursive
 */
rdomDoc.prototype.AddGeteRequest = function (path , recursive)
{
	/** @type {string} */
	var req;
	/** @type {string} */
	var mode;

	// Recursive?
	if (recursive === true)
		mode = 'R';
	else
		mode = 'N';

	// Create request.
	req = '<rdom fcn="gete" doc="' + this.docName + '" path="' + path + '" mode="' + mode + '"/>';

	// Clear the list and index.
	this.reqIdx = 0;
	this.reqList = [];

	// Add it to the list.
	this.reqList.push(req);
};

/**
 * @param {string} path
  */
rdomDoc.prototype.AddGetRequest = function (path)
{
	/** @type {string} */
	var req;

	// Create request.
	req = '<rdom fcn="get" doc="' + this.docName + '" path="' + path + '"/>';

	// Clear the list and index.
	this.reqIdx = 0;
	this.reqList = [];

	// Add it to the list.
	this.reqList.push(req);
};

/**
 * @param {string} path
 * @param {string} text
 */
rdomDoc.prototype.AddSetRequest = function (path , text)
{
	/** @type {string} */
	var req;

	// Create request.
	req = '<rdom fcn="set" doc="' + this.docName + '" path="' + path + '">' + text + '</rdom>';

	// Clear the list and index.
	this.reqIdx = 0;
	this.reqList = [];

	// Add it to the list.
	this.reqList.push(req);
};

/**
 * @param {Array.<string>} path_list
 */
rdomDoc.prototype.AddRequestList = function (path_list)
{
    /** @type {number} */
    var i;
    /** @type {string} */
    var req;
    /** @type {string} */
    var reqPre;
    /** @type {string} */
    var reqSeg;

    // Clear the path list and response.
    this.reqList = [];
    this.reqIdx = 0;
    this.response = null;
    // Create the request preamble.
    reqPre = '<rdom fcn="getl" doc="' + this.docName + '">';
    // Create initial request.
    req = reqPre;
    // For each element of path list.
    for (i = 0 ; i < path_list.length ; ++i)
    {
        // Create request segment.
        reqSeg = '<p>' + path_list[i] + '</p>';
        // If total size will exceed limit.
        if ((req.length + reqSeg.length) > MAX_REQ_SIZE)
        {
            // Finish request.
            req += '</rdom>';
            // Add to list.
            this.reqList.push(req);
            // Start new request.
            req = reqPre + reqSeg;
        }
        // Else there's room; add the segment.
        // Note that this could result in a 'req' that is EXACTLY
        // MAX_REQ_SIZE, So the resulting request, after adding the
        // trailing </rdom>, could exceed MAX_REQ_SIZE by 8 bytes.
        else
        {
            req += reqSeg;
        }
    }
    // Finish up.
    req += '</rdom>';
    // Add to list.
    this.reqList.push(req);
};

/**
 * Executes a 'gete' request to return an xml DOM tree.
 *
 * @param {string} path
 * @param {boolean} recursive
 * @param {function(rdomDoc) | null} callback
 */
rdomDoc.prototype.GetXmlDom = function (path, recursive, callback)
{
	var doc = new rdomDoc(this.docName, this.targetUrl);

	doc.async = this.async;
	doc.callback = callback;
	doc.AddGeteRequest(path, recursive);
	rdomQ.Add(doc);
};

/**
 * Executes a 'getl' request.
 *
 * @param {Array.<string>} path_list
 * @param {function(rdomDoc) | null} callback
 */
rdomDoc.prototype.GetList = function (path_list, callback)
{
	var doc = new rdomDoc(this.docName, this.targetUrl);

	doc.async = this.async;
	doc.callback = callback;
	doc.AddRequestList(path_list);
	rdomQ.Add(doc);
};

/**
 * Sends a 'get' request using an anonymous rdomDoc.
 *
 * @param {string} path
 * @param {function(rdomDoc) | null} callback
 */
rdomDoc.prototype.GetText = function (path, callback)
{
	var doc = new rdomDoc(this.docName, this.targetUrl);

	doc.async = this.async;
	doc.callback = callback;
	doc.AddGetRequest(path);
	rdomQ.Add(doc);
};

/**
 * Sends a 'set' request using an anonymous rdomDoc.
 *
 * @param {string} path
 * @param {string} text
 * @param {function(rdomDoc) | null} callback
 */
rdomDoc.prototype.SetText = function (path, text, callback)
{
	var doc = new rdomDoc(this.docName, this.targetUrl);

	doc.async = this.async;
	doc.callback = callback;
	doc.AddSetRequest(path, text);
	rdomQ.Add(doc);
};

/**
 * Change the read/write property of the file on the target.<br>
 *
 * @param {boolean} read_write
 * @param {function(rdomDoc) | null} callback
 */
rdomDoc.prototype.Mode = function (read_write , callback)
{
	/** @type {string} */
	var req;
	var doc = new rdomDoc(this.docName , this.targetUrl);

	// Create request to write element text.
	if (read_write === true)
		req = '<rdom fcn="mode" doc="' + this.docName + '" mode="W"/>';
	else
		req = '<rdom fcn="mode" doc="' + this.docName + '" mode="R"/>';

	doc.callback = callback;
	doc.reqIdx = 0;
	doc.reqList = [];
	doc.reqList.push(req);
	rdomQ.Add(doc);
};

/******************************************************************************
 *
 * RDOM Queue
 *
 * This class is used to sequence through submitted rdom requests.
 *
 ******************************************************************************/

/**
 * @constructor
 */
function rdomQConstructor()
{
    /** @type {Array.<rdomDoc>} */
    this.qList = [];
    /** @type {boolean} */
    this.running = false;

    /**
     * Adds an rdomDoc request to the list.
     *
     * @param {rdomDoc} doc
     */
    this.Add = function(doc)
    {
		// Send out directly if not async.
		if (!doc.async)
		{
			// We'll return from this when the request has finished.
			doc.Dispatch();
			// Return.
			return;
		}
		// If already running, or if the Q is not empty.
        if (this.running || (this.qList.length !== 0))
        {
            // Add it to the list.
            this.qList.push(doc);
            // And return.
            return;
        }
        //
        // If we get here, a request is currently not underway, and
        // the Q is empty.
        //
        // Set the flag.
        this.running = true;
        // Send it right away.
        doc.Dispatch();
    };

    /**
     * Called by the dispatch function when the last request
     * of an rdomDoc has been sent.
     */
    this.Done = function()
    {
        // If Q is not empty.
        if (this.qList.length !== 0)
        {
            // Send out the next request.
            this.qList.shift().Dispatch();
            // And return.
            return
        }
        // Q is empty, clear flag.
        this.running = false;
    };
}

/**
 * Instantiate the rdomQ.
 *
 * @type {rdomQConstructor}
 */
var rdomQ = new rdomQConstructor();
